<?php
declare(strict_types=1);
require __DIR__ . '/config.php';
session_start();
require __DIR__ . '/includes/auth.php';
require_login();

$userId = current_user_id();

$notice = '';
$error = '';

function upgrade_cost(int $base, float $mult, int $nextLevel): int {
  // cost for next level: base * mult^(nextLevel-1)
  return (int)round($base * pow($mult, max(0, $nextLevel - 1)));
}

// Buy upgrade
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['upgrade_id'])) {
  csrf_verify();
  $upgradeId = (int)($_POST['upgrade_id'] ?? 0);

  // Load upgrade + current level
  $stmt = $pdo->prepare("
    SELECT u.*, COALESCE(uu.level, 0) AS level
    FROM upgrades u
    LEFT JOIN user_upgrades uu ON uu.upgrade_id = u.id AND uu.user_id = ?
    WHERE u.id = ?
    LIMIT 1
  ");
  $stmt->execute([$userId, $upgradeId]);
  $up = $stmt->fetch();

  if (!$up) {
    $error = "Upgrade not found.";
  } else {
    $level = (int)$up['level'];
    $max = (int)$up['max_level'];
    if ($level >= $max) {
      $error = "Already max level.";
    } else {
      $nextLevel = $level + 1;
      $cost = upgrade_cost((int)$up['base_cost'], (float)$up['cost_multiplier'], $nextLevel);

      $stmt = $pdo->prepare("SELECT money FROM user_stats WHERE user_id = ?");
      $stmt->execute([$userId]);
      $money = (int)($stmt->fetch()['money'] ?? 0);

      if ($money < $cost) {
        $error = "Not enough money. Need £{$cost}.";
      } else {
        $pdo->beginTransaction();
        try {
          $stmt = $pdo->prepare("UPDATE user_stats SET money = money - ? WHERE user_id = ?");
          $stmt->execute([$cost, $userId]);

          $stmt = $pdo->prepare("
            INSERT INTO user_upgrades (user_id, upgrade_id, level)
            VALUES (?, ?, 1)
            ON DUPLICATE KEY UPDATE level = level + 1
          ");
          $stmt->execute([$userId, $upgradeId]);

          $stmt = $pdo->prepare("INSERT INTO activity_log (user_id, message) VALUES (?, ?)");
          $stmt->execute([$userId, "You upgraded " . $up['name'] . " to level {$nextLevel}."]);

          $pdo->commit();
          $notice = "Upgraded " . $up['name'] . " to level {$nextLevel}.";
        } catch (Throwable $e) {
          $pdo->rollBack();
          $error = "Upgrade failed. Try again.";
        }
      }
    }
  }
}

// Load upgrades list
$stmt = $pdo->prepare("
  SELECT u.*, COALESCE(uu.level, 0) AS level
  FROM upgrades u
  LEFT JOIN user_upgrades uu ON uu.upgrade_id = u.id AND uu.user_id = ?
  ORDER BY u.sort_order ASC, u.id ASC
");
$stmt->execute([$userId]);
$ups = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>Upgrades — State of Crime</title>
  <link rel="stylesheet" href="assets/css/app.css" />
</head>
<body>
  <?php include __DIR__ . '/includes/header.php'; ?>
  <div class="shell">
    <?php include __DIR__ . '/includes/sidebar.php'; ?>
    <main class="content">
      <section class="card">
        <h1 class="h1">Upgrades</h1>
        <p class="muted">Spend money to increase your success chances. Each level adds +1% (by default).</p>

        <?php if ($notice): ?><div class="ok" style="margin-top:12px;"><?= e($notice) ?></div><?php endif; ?>
        <?php if ($error): ?><div class="alert" style="margin-top:12px;"><?= e($error) ?></div><?php endif; ?>
      </section>

      <section class="grid" style="margin-top:14px;">
        <?php foreach ($ups as $u): ?>
          <?php
            $level = (int)$u['level'];
            $max = (int)$u['max_level'];
            $next = min($max, $level + 1);
            $cost = ($level >= $max) ? 0 : upgrade_cost((int)$u['base_cost'], (float)$u['cost_multiplier'], $next);
            $effect = ((int)$u['effect_per_level']) * $level;
            $applies = $u['applies_to'] === 'crime' ? 'Crimes' : 'Car Theft';
          ?>
          <div class="action" style="display:flex;flex-direction:column;gap:10px;">
            <div>
              <div class="action__title"><?= e($u['name']) ?> <span class="muted">(<?= $applies ?>)</span></div>
              <div class="action__desc"><?= e($u['description']) ?></div>
            </div>

            <div class="muted" style="display:flex;flex-wrap:wrap;gap:10px;font-size:.92rem;">
              <span>Level: <strong style="color:#fff;"><?= $level ?>/<?= $max ?></strong></span>
              <span>Bonus: <strong style="color:#fff;">+<?= $effect ?>%</strong></span>
              <?php if ($level < $max): ?>
                <span>Next cost: <strong style="color:#fff;">£<?= $cost ?></strong></span>
              <?php else: ?>
                <span><strong style="color:#fff;">Maxed</strong></span>
              <?php endif; ?>
            </div>

            <form method="post" style="margin-top:auto;">
              <input type="hidden" name="csrf_token" value="<?= e(csrf_token()) ?>">
              <input type="hidden" name="upgrade_id" value="<?= (int)$u['id'] ?>">
              <button class="btn" type="submit" <?= $level >= $max ? 'disabled' : '' ?> style="opacity:<?= $level >= $max ? '0.65' : '1' ?>;">
                <?= $level >= $max ? 'Max Level' : 'Buy Level ' . ($level + 1) ?>
              </button>
            </form>
          </div>
        <?php endforeach; ?>
      </section>
    </main>
    <?php include __DIR__ . '/includes/status_panel.php'; ?>
  </div>
  <script src="assets/js/app.js"></script>
</body>
</html>
