<?php
declare(strict_types=1);
require __DIR__ . '/config.php';
session_start();

if (!empty($_SESSION['user_id'])) {
  header('Location: dashboard.php');
  exit;
}

$error = '';
$ok = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $username = trim((string)($_POST['username'] ?? ''));
  $email = trim((string)($_POST['email'] ?? ''));
  $password = (string)($_POST['password'] ?? '');

  if ($username === '' || $email === '' || $password === '') {
    $error = 'All fields are required.';
  } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    $error = 'Enter a valid email.';
  } elseif (strlen($password) < 6) {
    $error = 'Password must be at least 6 characters.';
  } else {
    // uniqueness checks
    $stmt = $pdo->prepare("SELECT 1 FROM users WHERE username = ? OR email = ? LIMIT 1");
    $stmt->execute([$username, $email]);
    if ($stmt->fetch()) {
      $error = 'Username or email already taken.';
    } else {
      $hash = password_hash($password, PASSWORD_DEFAULT);
      $pdo->beginTransaction();
      try {
        // Create user
        $stmt = $pdo->prepare("INSERT INTO users (username, email, password_hash) VALUES (?, ?, ?)");
        $stmt->execute([$username, $email, $hash]);
        $newUserId = (int)$pdo->lastInsertId();

        // Pick default rank (Nobody)
        $rankStmt = $pdo->prepare("SELECT id FROM ranks WHERE name = ? LIMIT 1");
        $rankStmt->execute(['Nobody']);
        $rankId = (int)($rankStmt->fetch()['id'] ?? 0);
        if ($rankId <= 0) {
          // Fallback: first rank by sort_order
          $rankStmt = $pdo->query("SELECT id FROM ranks ORDER BY sort_order ASC, id ASC LIMIT 1");
          $rankId = (int)($rankStmt->fetch()['id'] ?? 0);
        }

        // Create default stats row
        $statsStmt = $pdo->prepare("
          INSERT INTO user_stats (user_id, money, bullets, points, health, max_health, rank_id)
          VALUES (?, 250, 0, 0, 100, 100, ?)
        ");
        $statsStmt->execute([$newUserId, $rankId > 0 ? $rankId : null]);

        $pdo->commit();
        $ok = 'Account created. You can now login.';
      } catch (Throwable $e) {
        $pdo->rollBack();
        throw $e;
      }
    }
  }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>Register — State of Crime</title>
  <link rel="stylesheet" href="assets/css/app.css" />
</head>
<body class="auth">
  <div class="authcard">
    <div class="authbrand">STATE OF CRIME</div>
    <div class="muted">Create your account</div>

    <?php if ($error): ?>
      <div class="alert"><?= htmlspecialchars($error) ?></div>
    <?php endif; ?>
    <?php if ($ok): ?>
      <div class="ok"><?= htmlspecialchars($ok) ?></div>
    <?php endif; ?>

    <form method="post" class="authform">
      <label class="field">
        <span>Username</span>
        <input name="username" required />
      </label>
      <label class="field">
        <span>Email</span>
        <input type="email" name="email" required />
      </label>
      <label class="field">
        <span>Password</span>
        <input type="password" name="password" required />
      </label>
      <button class="btn" type="submit">Create account</button>
    </form>

    <div class="authlinks">
      <a href="login.php">Back to login</a>
    </div>
  </div>
</body>
</html>
