<?php
declare(strict_types=1);
require __DIR__ . '/config.php';
session_start();
require __DIR__ . '/includes/auth.php';
require_login();

$viewerId = current_user_id();

$targetId = (int)($_GET['u'] ?? $viewerId);
if ($targetId <= 0) $targetId = (int)$viewerId;

$stmt = $pdo->prepare("
  SELECT u.id, u.username, u.bio, u.created_at, u.last_active,
         s.money, s.bullets, s.points, s.health, s.max_health,
         COALESCE(r.name, 'Unknown') AS rank_name
  FROM users u
  JOIN user_stats s ON s.user_id = u.id
  LEFT JOIN ranks r ON r.id = s.rank_id
  WHERE u.id = ?
  LIMIT 1
");
$stmt->execute([$targetId]);
$user = $stmt->fetch();

if (!$user) {
  http_response_code(404);
  echo "User not found.";
  exit;
}

$isMe = ((int)$user['id'] === (int)$viewerId);

$notice = '';
$error = '';
if ($isMe && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['bio'])) {
  csrf_verify();
  $bio = trim((string)($_POST['bio'] ?? ''));
  if (mb_strlen($bio) > 255) {
    $error = "Bio must be 255 characters or less.";
  } else {
    $stmt = $pdo->prepare("UPDATE users SET bio = ? WHERE id = ?");
    $stmt->execute([$bio, $viewerId]);
    $notice = "Bio updated.";
    $user['bio'] = $bio;
  }
}

function fmt_date(string $ts): string {
  return date("d M Y", strtotime($ts));
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>Profile — State of Crime</title>
  <link rel="stylesheet" href="assets/css/app.css" />
</head>
<body>
  <?php include __DIR__ . '/includes/header.php'; ?>

  <div class="shell">
    <?php include __DIR__ . '/includes/sidebar.php'; ?>

    <main class="content">
      <section class="card">
        <h1 class="h1"><?= e($user['username']) ?> <span class="muted">(<?= e($user['rank_name']) ?>)</span></h1>
        <p class="muted">Joined: <?= e(fmt_date($user['created_at'])) ?></p>

        <?php if ($notice): ?><div class="ok" style="margin-top:12px;"><?= e($notice) ?></div><?php endif; ?>
        <?php if ($error): ?><div class="alert" style="margin-top:12px;"><?= e($error) ?></div><?php endif; ?>

        <div class="twocol" style="margin-top:14px;">
          <div class="card" style="padding:14px;">
            <h2 class="h2">Stats</h2>
            <div class="muted" style="display:flex;flex-direction:column;gap:8px;">
              <div class="item"><span>Points</span><em><?= (int)$user['points'] ?></em></div>
              <div class="item"><span>Money</span><em>£<?= number_format((int)$user['money']) ?></em></div>
              <div class="item"><span>Bullets</span><em><?= (int)$user['bullets'] ?></em></div>
              <div class="item"><span>Health</span><em><?= (int)$user['health'] ?> / <?= (int)$user['max_health'] ?></em></div>
            </div>
          </div>

          <div class="card" style="padding:14px;">
            <h2 class="h2">Bio</h2>
            <?php if ($isMe): ?>
              <form method="post" style="margin-top:10px;">
                <input type="hidden" name="csrf_token" value="<?= e(csrf_token()) ?>">
                <label class="field">
                  <span class="muted">Short bio (max 255 chars)</span>
                  <input name="bio" value="<?= e((string)$user['bio']) ?>" maxlength="255" />
                </label>
                <button class="btn" type="submit" style="margin-top:10px;">Save Bio</button>
              </form>
            <?php else: ?>
              <p class="muted" style="margin-top:10px;"><?= e((string)$user['bio'] ?: 'No bio set.') ?></p>
              <div style="margin-top:12px;">
                <a class="btn" href="inbox.php">Message (coming next)</a>
              </div>
            <?php endif; ?>
          </div>
        </div>
      </section>
    </main>

    <?php include __DIR__ . '/includes/status_panel.php'; ?>
  </div>

  <script src="assets/js/app.js"></script>
</body>
</html>
