<?php
declare(strict_types=1);
require __DIR__ . '/config.php';
session_start();
require __DIR__ . '/includes/auth.php';
require_login();

$userId = current_user_id();

// Auto-redirect
$stmt = $pdo->prepare("SELECT jail_until, hospital_until FROM user_stats WHERE user_id = ? LIMIT 1");
$stmt->execute([$userId]);
$r = $stmt->fetch() ?: ['jail_until'=>null,'hospital_until'=>null];
$jLeft = $r['jail_until'] ? max(0, strtotime($r['jail_until']) - time()) : 0;
$hLeft = $r['hospital_until'] ? max(0, strtotime($r['hospital_until']) - time()) : 0;
if ($jLeft > 0) { header('Location: jail.php'); exit; }
if ($hLeft > 0) { header('Location: hospital.php'); exit; }

$notice = '';
$error = '';

function seconds_left(?string $t): int {
  if (!$t) return 0;
  return max(0, strtotime($t) - time());
}
function fmt(int $s): string {
  if ($s <= 0) return "0s";
  $m = intdiv($s, 60);
  $r = $s % 60;
  if ($m > 0) return $m . "m " . str_pad((string)$r, 2, "0", STR_PAD_LEFT) . "s";
  return $r . "s";
}

// Resolve any runs that should complete (MVP: resolved on page load/poll)
function maybe_resolve_runs(PDO $pdo): void {
  // Find runs that are pending and start_at passed, and not resolved yet
  $stmt = $pdo->query("
    SELECT id, organised_crime_id
    FROM organised_crime_runs
    WHERE status IN ('pending','running') AND start_at IS NOT NULL AND start_at <= NOW() AND resolved_at IS NULL
    ORDER BY start_at ASC
    LIMIT 10
  ");
  $runs = $stmt->fetchAll();
  foreach ($runs as $run) {
    $runId = (int)$run['id'];
    $crimeId = (int)$run['organised_crime_id'];

    $pdo->beginTransaction();
    try {
      // lock run
      $lock = $pdo->prepare("SELECT * FROM organised_crime_runs WHERE id = ? FOR UPDATE");
      $lock->execute([$runId]);
      $rrow = $lock->fetch();
      if (!$rrow || $rrow['resolved_at'] !== null) { $pdo->rollBack(); continue; }

      // load crime
      $cstmt = $pdo->prepare("SELECT * FROM organised_crimes WHERE id = ?");
      $cstmt->execute([$crimeId]);
      $crime = $cstmt->fetch();
      if (!$crime) { $pdo->rollBack(); continue; }

      // members
      $mstmt = $pdo->prepare("
        SELECT m.user_id, s.points, COALESCE(r.rank_bonus,0) AS rank_bonus
        FROM organised_crime_members m
        JOIN user_stats s ON s.user_id = m.user_id
        LEFT JOIN ranks r ON r.id = s.rank_id
        WHERE m.run_id = ?
      ");
      $mstmt->execute([$runId]);
      $members = $mstmt->fetchAll();

      if (count($members) < (int)$crime['required_members']) {
        // Not enough members; keep pending but push start_at forward slightly
        $upd = $pdo->prepare("UPDATE organised_crime_runs SET start_at = DATE_ADD(NOW(), INTERVAL 15 SECOND), status='pending' WHERE id = ?");
        $upd->execute([$runId]);
        $pdo->commit();
        continue;
      }

      // average bonuses
      $avgRank = (int)round(array_sum(array_map(fn($x)=>(int)$x['rank_bonus'], $members)) / max(1,count($members)));
      $avgPoints = (int)round(array_sum(array_map(fn($x)=>(int)$x['points'], $members)) / max(1,count($members)));
      $pointsBonus = min(10, intdiv(max(0, $avgPoints), 25));

      // upgrade bonus (crime upgrades) average across members
      $uids = array_map(fn($x)=>(int)$x['user_id'], $members);
      $place = implode(',', array_fill(0, count($uids), '?'));
      $ustmt = $pdo->prepare("
        SELECT COALESCE(AVG(t.bonus), 0) AS avg_bonus FROM (
          SELECT uu.user_id, SUM(uu.level * u.effect_per_level) AS bonus
          FROM upgrades u
          JOIN user_upgrades uu ON uu.upgrade_id = u.id
          WHERE u.applies_to='crime' AND uu.user_id IN ($place)
          GROUP BY uu.user_id
        ) t
      ");
      $ustmt->execute($uids);
      $avgUpgrade = (int)round((float)($ustmt->fetch()['avg_bonus'] ?? 0));

      $base = (int)$crime['base_success_rate'];
      $effective = max(5, min(95, $base + $avgRank + $pointsBonus + $avgUpgrade));

      $roll = random_int(1,100);
      $success = $roll <= $effective;

      $totalReward = random_int((int)$crime['min_total_reward'], (int)$crime['max_total_reward']);
      $totalPoints = random_int((int)$crime['min_total_points'], (int)$crime['max_total_points']);

      $perReward = (int)floor($totalReward / count($members));
      $perPoints = (int)floor($totalPoints / count($members));

      // Apply results + cooldown + consequences
      foreach ($members as $m) {
        $uid = (int)$m['user_id'];
        // cooldown
        $cd = $pdo->prepare("
          INSERT INTO user_organised_crime_cooldowns (user_id, organised_crime_id, last_attempt_at)
          VALUES (?, ?, NOW())
          ON DUPLICATE KEY UPDATE last_attempt_at = NOW()
        ");
        $cd->execute([$uid, $crimeId]);

        if ($success) {
          $upd = $pdo->prepare("UPDATE user_stats SET money = money + ?, points = points + ? WHERE user_id = ?");
          $upd->execute([$perReward, $perPoints, $uid]);
          $msg = sprintf("Organised crime: %s — Success (+£%d, +%d points).", $crime['name'], $perReward, $perPoints);
        } else {
          // failure: jail or hospital chance (prefer jail if hit)
          $goJail = random_int(1,100) <= (int)$crime['jail_chance'];
          $goHosp = (!$goJail) && (random_int(1,100) <= (int)$crime['hospital_chance']);

          if ($goJail) {
            $upd = $pdo->prepare("UPDATE user_stats SET jail_until = DATE_ADD(NOW(), INTERVAL ? SECOND) WHERE user_id = ?");
            $upd->execute([(int)$crime['jail_seconds'], $uid]);
            $msg = sprintf("Organised crime: %s — Failed and got arrested (Jail %ds).", $crime['name'], (int)$crime['jail_seconds']);
          } elseif ($goHosp) {
            $upd = $pdo->prepare("UPDATE user_stats SET hospital_until = DATE_ADD(NOW(), INTERVAL ? SECOND) WHERE user_id = ?");
            $upd->execute([(int)$crime['hospital_seconds'], $uid]);
            $msg = sprintf("Organised crime: %s — Failed and got injured (Hospital %ds).", $crime['name'], (int)$crime['hospital_seconds']);
          } else {
            $msg = sprintf("Organised crime: %s — Failed. You slipped away.", $crime['name']);
          }
        }

        $log = $pdo->prepare("INSERT INTO activity_log (user_id, message) VALUES (?, ?)");
        $log->execute([$uid, $msg]);
      }

      $updRun = $pdo->prepare("
        UPDATE organised_crime_runs
        SET status='completed', resolved_at = NOW(), outcome = ?, total_reward = ?, total_points = ?
        WHERE id = ?
      ");
      $updRun->execute([$success ? 'success' : 'fail', $totalReward, $totalPoints, $runId]);

      $pdo->commit();
    } catch (Throwable $e) {
      $pdo->rollBack();
    }
  }
}

maybe_resolve_runs($pdo);

// Join/leave actions (non-AJAX MVP)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['crime_id'])) {
  csrf_verify();
  $crimeId = (int)($_POST['crime_id'] ?? 0);
  $action = (string)($_POST['action'] ?? 'join');

  // Cooldown check
  $stmt = $pdo->prepare("
    SELECT oc.cooldown_seconds, uocc.last_attempt_at
    FROM organised_crimes oc
    LEFT JOIN user_organised_crime_cooldowns uocc
      ON uocc.organised_crime_id = oc.id AND uocc.user_id = ?
    WHERE oc.id = ?
    LIMIT 1
  ");
  $stmt->execute([$userId, $crimeId]);
  $cd = $stmt->fetch();
  if (!$cd) {
    $error = "Organised crime not found.";
  } else {
    $last = $cd['last_attempt_at'] ?? null;
    if ($last) {
      $left = max(0, strtotime($last) + (int)$cd['cooldown_seconds'] - time());
      if ($left > 0) $error = "You're on cooldown for " . fmt($left) . ".";
    }
  }

  if ($error === '') {
    $pdo->beginTransaction();
    try {
      // Find or create pending run for this crime (one at a time)
      $stmt = $pdo->prepare("SELECT id, status, start_at FROM organised_crime_runs WHERE organised_crime_id = ? AND status = 'pending' LIMIT 1 FOR UPDATE");
      $stmt->execute([$crimeId]);
      $run = $stmt->fetch();

      if (!$run) {
        $ins = $pdo->prepare("INSERT INTO organised_crime_runs (organised_crime_id, status) VALUES (?, 'pending')");
        $ins->execute([$crimeId]);
        $runId = (int)$pdo->lastInsertId();
        $run = ['id'=>$runId, 'status'=>'pending', 'start_at'=>null];
      } else {
        $runId = (int)$run['id'];
      }

      if ($action === 'leave') {
        $del = $pdo->prepare("DELETE FROM organised_crime_members WHERE run_id = ? AND user_id = ?");
        $del->execute([$runId, $userId]);
        $pdo->commit();
        header("Location: organised_crime.php");
        exit;
      }

      // join
      $ins = $pdo->prepare("
        INSERT INTO organised_crime_members (run_id, user_id)
        VALUES (?, ?)
        ON DUPLICATE KEY UPDATE joined_at = joined_at
      ");
      $ins->execute([$runId, $userId]);

      // Check member count and if full, schedule countdown if not scheduled yet
      $cstmt = $pdo->prepare("SELECT required_members, countdown_seconds FROM organised_crimes WHERE id = ?");
      $cstmt->execute([$crimeId]);
      $crime = $cstmt->fetch();
      $req = (int)$crime['required_members'];

      $mstmt = $pdo->prepare("SELECT COUNT(*) AS c FROM organised_crime_members WHERE run_id = ?");
      $mstmt->execute([$runId]);
      $count = (int)($mstmt->fetch()['c'] ?? 0);

      if ($count >= $req) {
        // set start_at if not already set
        $upd = $pdo->prepare("UPDATE organised_crime_runs SET start_at = COALESCE(start_at, DATE_ADD(NOW(), INTERVAL ? SECOND)), status='running' WHERE id = ?");
        $upd->execute([(int)$crime['countdown_seconds'], $runId]);
      }

      $pdo->commit();
      header("Location: organised_crime.php");
      exit;
    } catch (Throwable $e) {
      $pdo->rollBack();
      $error = "Could not join right now. Try again.";
    }
  }
}

// Load list + active run state for each crime
$crimes = $pdo->query("SELECT * FROM organised_crimes ORDER BY sort_order ASC, id ASC")->fetchAll();

// Find pending/running run and member counts
$runInfo = [];
foreach ($crimes as $c) {
  $cid = (int)$c['id'];
  $stmt = $pdo->prepare("SELECT id, status, start_at FROM organised_crime_runs WHERE organised_crime_id = ? AND status IN ('pending','running') ORDER BY id DESC LIMIT 1");
  $stmt->execute([$cid]);
  $run = $stmt->fetch();
  if ($run) {
    $rid = (int)$run['id'];
    $mstmt = $pdo->prepare("SELECT COUNT(*) AS c FROM organised_crime_members WHERE run_id = ?");
    $mstmt->execute([$rid]);
    $count = (int)($mstmt->fetch()['c'] ?? 0);

    $meIn = $pdo->prepare("SELECT 1 FROM organised_crime_members WHERE run_id = ? AND user_id = ? LIMIT 1");
    $meIn->execute([$rid, $userId]);
    $in = (bool)$meIn->fetch();

    $runInfo[$cid] = [
      'run_id'=>$rid,
      'status'=>$run['status'],
      'start_at'=>$run['start_at'],
      'count'=>$count,
      'in'=>$in
    ];
  } else {
    $runInfo[$cid] = null;
  }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>Organised Crime — State of Crime</title>
  <link rel="stylesheet" href="assets/css/app.css" />
</head>
<body>
  <?php include __DIR__ . '/includes/header.php'; ?>
  <div class="shell">
    <?php include __DIR__ . '/includes/sidebar.php'; ?>
    <main class="content">
      <section class="card">
        <h1 class="h1">Organised Crime</h1>
        <p class="muted">Join a crew. When it's full, a countdown starts and the job resolves automatically.</p>
        <?php if ($error): ?><div class="alert" style="margin-top:12px;"><?= e($error) ?></div><?php endif; ?>
        <?php if ($notice): ?><div class="ok" style="margin-top:12px;"><?= e($notice) ?></div><?php endif; ?>
      </section>

      <section class="grid" style="margin-top:14px;">
        <?php foreach ($crimes as $c): ?>
          <?php
            $cid = (int)$c['id'];
            $info = $runInfo[$cid] ?? null;
            $req = (int)$c['required_members'];
            $count = $info ? (int)$info['count'] : 0;
            $in = $info ? (bool)$info['in'] : false;
            $startLeft = $info && $info['start_at'] ? seconds_left($info['start_at']) : 0;
            $status = $info ? $info['status'] : 'idle';
          ?>
          <div class="action" style="display:flex;flex-direction:column;gap:10px;">
            <div>
              <div class="action__title"><?= e($c['name']) ?></div>
              <div class="action__desc"><?= e($c['description']) ?></div>
            </div>

            <div class="muted" style="display:flex;flex-wrap:wrap;gap:10px;font-size:.92rem;">
              <span>Crew: <strong style="color:#fff;"><?= $count ?>/<?= $req ?></strong></span>
              <span>Base success: <strong style="color:#fff;"><?= (int)$c['base_success_rate'] ?>%</strong></span>
              <span>Total reward: <strong style="color:#fff;">£<?= (int)$c['min_total_reward'] ?>–£<?= (int)$c['max_total_reward'] ?></strong></span>
              <span>Status: <strong style="color:#fff;"><?= e($status) ?></strong></span>
              <?php if ($status === 'running' && $startLeft > 0): ?>
                <span>Starts in: <strong style="color:#fff;"><?= e(fmt($startLeft)) ?></strong></span>
              <?php endif; ?>
            </div>

            <form method="post" style="margin-top:auto;">
              <input type="hidden" name="csrf_token" value="<?= e(csrf_token()) ?>">
              <input type="hidden" name="crime_id" value="<?= $cid ?>">
              <?php if ($in): ?>
                <input type="hidden" name="action" value="leave">
                <button class="btn" type="submit" <?= ($status === 'running' && $startLeft > 0) ? 'disabled' : '' ?> style="opacity:<?= ($status === 'running' && $startLeft > 0) ? '0.65' : '1' ?>;">
                  <?= ($status === 'running' && $startLeft > 0) ? 'Locked In' : 'Leave Crew' ?>
                </button>
              <?php else: ?>
                <input type="hidden" name="action" value="join">
                <button class="btn" type="submit">Join Crew</button>
              <?php endif; ?>
            </form>
          </div>
        <?php endforeach; ?>
      </section>
    </main>
    <?php include __DIR__ . '/includes/status_panel.php'; ?>
  </div>
  <script src="assets/js/app.js"></script>
  <script>
    // Light auto-refresh so countdowns resolve without manual refresh
    setInterval(()=>{ window.location.reload(); }, 15000);
  </script>
</body>
</html>
