<?php
declare(strict_types=1);
require __DIR__ . '/config.php';
session_start();
require __DIR__ . '/includes/auth.php';
require_login();

// AUTO_REDIRECT_BLOCK
if ($jLeft > 0) { header('Location: jail.php'); exit; }
if ($hLeft > 0) { header('Location: hospital.php'); exit; }
// Auto-redirect if in jail/hospital (quality-of-life)
$stmt = $pdo->prepare("SELECT jail_until, hospital_until FROM user_stats WHERE user_id = ? LIMIT 1");
$stmt->execute([$userId]);
$_restr = $stmt->fetch() ?: ['jail_until'=>null,'hospital_until'=>null];
$jLeft = $_restr['jail_until'] ? max(0, strtotime($_restr['jail_until']) - time()) : 0;
$hLeft = $_restr['hospital_until'] ? max(0, strtotime($_restr['hospital_until']) - time()) : 0;


$userId = current_user_id();

// Fetch basic user (for greeting).
$stmt = $pdo->prepare("SELECT username FROM users WHERE id = ?");
$stmt->execute([$userId]);
$me = $stmt->fetch() ?: ['username' => 'Player'];

// Fetch player stats used for crime modifiers (points, rank bonus, jail)
$stmt = $pdo->prepare("
  SELECT s.points, s.jail_until, s.hospital_until, COALESCE(r.rank_bonus, 0) AS rank_bonus
  FROM user_stats s
  LEFT JOIN ranks r ON r.id = s.rank_id
  WHERE s.user_id = ?
  LIMIT 1
");
$stmt->execute([$userId]);
$stats = $stmt->fetch() ?: ['points' => 0, 'jail_until' => null, 'hospital_until' => null, 'rank_bonus' => 0];

$stmt = $pdo->prepare("
  SELECT COALESCE(SUM(uu.level * u.effect_per_level), 0) AS bonus
  FROM upgrades u
  JOIN user_upgrades uu ON uu.upgrade_id = u.id
  WHERE uu.user_id = ? AND u.applies_to = 'crime'
");
$stmt->execute([$userId]);
$upgradeBonus = (int)($stmt->fetch()['bonus'] ?? 0);


function jail_seconds_left(?string $until): int {
  if (!$until) return 0;
  return max(0, strtotime($until) - time());
}

function hospital_seconds_left(?string $until): int {
  if (!$until) return 0;
  return max(0, strtotime($until) - time());
}
  if (!$until) return 0;
  return max(0, strtotime($until) - time());
}

$notice = '';
$error = '';

function seconds_left(string $lastAttemptAt, int $cooldownSeconds): int {
  $last = strtotime($lastAttemptAt);
  $next = $last + $cooldownSeconds;
  return max(0, $next - time());
}

function fmt_time(int $s): string {
  if ($s <= 0) return "Ready";
  $m = intdiv($s, 60);
  $r = $s % 60;
  if ($m > 0) return $m . "m " . str_pad((string)$r, 2, "0", STR_PAD_LEFT) . "s";
  return $r . "s";
}

// Handle crime attempt
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['crime_id'])) {
  // Block crimes while in jail
  $jLeft = jail_seconds_left($stats['jail_until']);
  if ($jLeft > 0) {
    $error = "You're in jail for " . fmt_time($jLeft) . ".";
  } elseif ($hLeft > 0) {
    $error = "You're in hospital for " . fmt_time($hLeft) . ".";
  }

  csrf_verify();

  $crimeId = (int)($_POST['crime_id'] ?? 0);

  // Load crime definition
  $stmt = $pdo->prepare("SELECT * FROM crimes WHERE id = ? LIMIT 1");
  $stmt->execute([$crimeId]);
  $crime = $stmt->fetch();

  if (!$crime) {
    $error = "Crime not found.";
  } else {
    // Check cooldown
    $stmt = $pdo->prepare("SELECT last_attempt_at FROM user_crime_cooldowns WHERE user_id = ? AND crime_id = ? LIMIT 1");
    $stmt->execute([$userId, $crimeId]);
    $cool = $stmt->fetch();

    if ($cool) {
      $left = seconds_left($cool['last_attempt_at'], (int)$crime['cooldown_seconds']);
      if ($left > 0) {
        $error = "You're laying low. Try again in " . fmt_time($left) . ".";
      }
    }

    if ($error === '') {
      // Roll
      $roll = random_int(1, 100);
      $baseRate = (int)$crime['success_rate'];
      $rankBonus = (int)($stats['rank_bonus'] ?? 0);
      $pointsNow = (int)($stats['points'] ?? 0);
      // Points bonus: +1% per 25 points, capped at +10%
      $pointsBonus = min(10, intdiv(max(0, $pointsNow), 25));
      $effectiveRate = max(5, min(95, $baseRate + $rankBonus + $pointsBonus + $upgradeBonus));

      $isSuccess = $roll <= $effectiveRate;

      $reward = 0;
      $points = 0;
      $damage = 0;

      if ($isSuccess) {
        $reward = random_int((int)$crime['min_reward'], (int)$crime['max_reward']);
        $points = random_int((int)$crime['min_points'], (int)$crime['max_points']);
      } else {
        // Failure consequence: small health loss
        $damage = random_int(2, 10);
        // Chance to go to jail on failure
        $jailRoll = random_int(1, 100);
        $jailChance = (int)$crime['jail_chance'];
        $jailSeconds = (int)$crime['jail_seconds'];
        $goJail = $jailRoll <= $jailChance;
      }

      // Apply results atomically
      $pdo->beginTransaction();
      try {
        // Upsert cooldown
        $stmt = $pdo->prepare("
          INSERT INTO user_crime_cooldowns (user_id, crime_id, last_attempt_at)
          VALUES (?, ?, NOW())
          ON DUPLICATE KEY UPDATE last_attempt_at = NOW()
        ");
        $stmt->execute([$userId, $crimeId]);

        if ($isSuccess) {
          $stmt = $pdo->prepare("
            UPDATE user_stats
            SET money = money + ?, points = points + ?
            WHERE user_id = ?
          ");
          $stmt->execute([$reward, $points, $userId]);

          $logMsg = sprintf("You committed %s — Success (+£%d, +%d points).", $crime['name'], $reward, $points);
          $notice = $logMsg;
        } else {
          $stmt = $pdo->prepare("
            UPDATE user_stats
            SET health = GREATEST(0, health - ?)
            WHERE user_id = ?
          ");
          $stmt->execute([$damage, $userId]);

          if (!empty($goJail)) {
            $stmt = $pdo->prepare("UPDATE user_stats SET jail_until = DATE_ADD(NOW(), INTERVAL ? SECOND) WHERE user_id = ?");
            $stmt->execute([$jailSeconds, $userId]);
          }

          $logMsg = !empty($goJail)
            ? sprintf("You attempted %s — Failed (-%d health) and got arrested (Jail %ds).", $crime['name'], $damage, $jailSeconds)
            : sprintf("You attempted %s — Failed (-%d health).", $crime['name'], $damage);
          $error = $logMsg; // show as error for feedback
        }

        $stmt = $pdo->prepare("INSERT INTO activity_log (user_id, message) VALUES (?, ?)");
        $stmt->execute([$userId, $logMsg]);

        $pdo->commit();
      } catch (Throwable $e) {
        $pdo->rollBack();
        $error = "Something went wrong. Please try again.";
      }
    }
  }
}

// Load crimes + cooldown info
$stmt = $pdo->query("SELECT * FROM crimes ORDER BY sort_order ASC, id ASC");
$crimes = $stmt->fetchAll();

// Get cooldowns for this user keyed by crime_id
$stmt = $pdo->prepare("SELECT crime_id, last_attempt_at FROM user_crime_cooldowns WHERE user_id = ?");
$stmt->execute([$userId]);
$cooldowns = [];
foreach ($stmt->fetchAll() as $row) {
  $cooldowns[(int)$row['crime_id']] = $row['last_attempt_at'];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>Crimes — State of Crime</title>
  <link rel="stylesheet" href="assets/css/app.css" />
</head>
<body>
  <?php include __DIR__ . '/includes/header.php'; ?>

  <div class="shell">
    <?php include __DIR__ . '/includes/sidebar.php'; ?>

    <main class="content">
      <?php $jLeft = jail_seconds_left($stats['jail_until'] ?? null); $hLeft = hospital_seconds_left($stats['hospital_until'] ?? null); ?>
      <section class="card">
        <h1 class="h1">Crimes</h1>
        <p class="muted">Choose a job. Success is affected by your rank and points. Lay low during cooldowns. Your status panel will update automatically.</p>
        <p class="muted" style="margin-top:8px;">Your modifiers: <strong style="color:#fff;">Rank +<?= (int)($stats['rank_bonus'] ?? 0) ?>%</strong> • <strong style="color:#fff;">Points +<?= min(10, intdiv(max(0, (int)($stats['points'] ?? 0)), 25)) ?>%</strong> • <strong style="color:#fff;">Street Smarts +<?= (int)$upgradeBonus ?>%</strong></p>

        <?php if ($jLeft > 0): ?>
          <div class="alert" style="margin-top:12px;">You can use options on the <a href="jail.php" style="color:#fff;">Jail</a> page.</div>
        <?php endif; ?>
        <?php if ($hLeft > 0): ?>
          <div class="alert" style="margin-top:12px;">You are currently in hospital for <?= e(fmt_time($hLeft)) ?>. Crimes are disabled. <a href="hospital.php" style="color:#fff;">Go to Hospital</a></div>
        <?php endif; ?>

        <?php if ($notice): ?>
          <div class="alert" style="margin-top:12px;">You are currently in jail for <?= e(fmt_time($jLeft)) ?>. Crimes are disabled.</div>
        <?php endif; ?>

        <?php if ($notice): ?>
          <div class="ok" style="margin-top:12px;"><?= e($notice) ?></div>
        <?php endif; ?>
        <?php if ($error): ?>
          <div class="alert" style="margin-top:12px;"><?= e($error) ?></div>
        <?php endif; ?>
      </section>

      <section class="grid" style="margin-top:14px;">
        <?php foreach ($crimes as $c): ?>
          <?php
            $cid = (int)$c['id'];
            $last = $cooldowns[$cid] ?? null;
            $left = $last ? seconds_left($last, (int)$c['cooldown_seconds']) : 0;
            $disabled = ($left > 0) || ($jLeft > 0) || ($hLeft > 0);
          ?>
          <div class="action" style="display:flex;flex-direction:column;gap:10px;">
            <div>
              <div class="action__title"><?= e($c['name']) ?></div>
              <div class="action__desc"><?= e($c['description']) ?></div>
            </div>

            <div class="muted" style="display:flex;flex-wrap:wrap;gap:10px;font-size:.92rem;">
              <?php $effective = max(5, min(95, (int)$c['success_rate'] + (int)($stats['rank_bonus'] ?? 0) + min(10, intdiv(max(0, (int)($stats['points'] ?? 0)), 25)) + (int)$upgradeBonus)); ?>
              <span>Success: <strong style="color:#fff;"><?= (int)$effective ?>%</strong></span>
              <span>Reward: <strong style="color:#fff;">£<?= (int)$c['min_reward'] ?>–£<?= (int)$c['max_reward'] ?></strong></span>
              <span>Cooldown: <strong style="color:#fff;"><?= (int)$c['cooldown_seconds'] ?>s</strong></span>
              <span>Jail: <strong style="color:#fff;"><?= (int)$c['jail_chance'] ?>% / <?= (int)$c['jail_seconds'] ?>s</strong></span>
            </div>

            <form method="post" style="margin-top:auto;">
              <input type="hidden" name="csrf_token" value="<?= e(csrf_token()) ?>">
              <input type="hidden" name="crime_id" value="<?= $cid ?>">
              <button class="btn" type="submit" <?= $disabled ? 'disabled' : '' ?> style="opacity:<?= $disabled ? '0.65' : '1' ?>;">
                <?= ($jLeft > 0) ? ('In Jail: ' . e(fmt_time($jLeft))) : (($hLeft > 0) ? ('In Hospital: ' . e(fmt_time($hLeft))) : ($disabled ? ('Cooldown: ' . e(fmt_time($left))) : 'Commit Crime')) ?>
              </button>
            </form>
          </div>
        <?php endforeach; ?>
      </section>
    </main>

    <?php include __DIR__ . '/includes/status_panel.php'; ?>
  </div>

  <script src="assets/js/app.js"></script>
</body>
</html>
