<?php
declare(strict_types=1);
require __DIR__ . '/config.php';
session_start();
require __DIR__ . '/includes/auth.php';
require_login();

// AUTO_REDIRECT_BLOCK
if ($jLeft > 0) { header('Location: jail.php'); exit; }
if ($hLeft > 0) { header('Location: hospital.php'); exit; }
// Auto-redirect if in jail/hospital (quality-of-life)
$stmt = $pdo->prepare("SELECT jail_until, hospital_until FROM user_stats WHERE user_id = ? LIMIT 1");
$stmt->execute([$userId]);
$_restr = $stmt->fetch() ?: ['jail_until'=>null,'hospital_until'=>null];
$jLeft = $_restr['jail_until'] ? max(0, strtotime($_restr['jail_until']) - time()) : 0;
$hLeft = $_restr['hospital_until'] ? max(0, strtotime($_restr['hospital_until']) - time()) : 0;


$userId = current_user_id();

$notice = '';
$error = '';

function seconds_left(string $lastAttemptAt, int $cooldownSeconds): int {
  $last = strtotime($lastAttemptAt);
  $next = $last + $cooldownSeconds;
  return max(0, $next - time());
}
function fmt_time(int $s): string {
  if ($s <= 0) return "Ready";
  $m = intdiv($s, 60);
  $r = $s % 60;
  if ($m > 0) return $m . "m " . str_pad((string)$r, 2, "0", STR_PAD_LEFT) . "s";
  return $r . "s";
}
function until_left(?string $until): int {
  if (!$until) return 0;
  return max(0, strtotime($until) - time());
}

// Player modifiers
$stmt = $pdo->prepare("
  SELECT s.points, s.jail_until, s.hospital_until, COALESCE(r.rank_bonus, 0) AS rank_bonus
  FROM user_stats s
  LEFT JOIN ranks r ON r.id = s.rank_id
  WHERE s.user_id = ?
  LIMIT 1
");
$stmt->execute([$userId]);
$stats = $stmt->fetch() ?: ['points' => 0, 'jail_until' => null, 'hospital_until' => null, 'rank_bonus' => 0];

// Upgrade bonus for car theft
$stmt = $pdo->prepare("
  SELECT COALESCE(SUM(uu.level * u.effect_per_level), 0) AS bonus
  FROM upgrades u
  JOIN user_upgrades uu ON uu.upgrade_id = u.id
  WHERE uu.user_id = ? AND u.applies_to = 'car_theft'
");
$stmt->execute([$userId]);
$upgradeBonus = (int)($stmt->fetch()['bonus'] ?? 0);

$jLeft = until_left($stats['jail_until'] ?? null);
$hLeft = until_left($stats['hospital_until'] ?? null);

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['car_theft_id'])) {
  csrf_verify();

  if ($jLeft > 0) {
    $error = "You're in jail for " . fmt_time($jLeft) . ".";
  } elseif ($hLeft > 0) {
    $error = "You're in hospital for " . fmt_time($hLeft) . ".";
  } else {
    $ctId = (int)($_POST['car_theft_id'] ?? 0);
    $stmt = $pdo->prepare("SELECT * FROM car_thefts WHERE id = ? LIMIT 1");
    $stmt->execute([$ctId]);
    $ct = $stmt->fetch();

    if (!$ct) {
      $error = "Car theft option not found.";
    } else {
      // Check cooldown
      $stmt = $pdo->prepare("SELECT last_attempt_at FROM user_car_theft_cooldowns WHERE user_id = ? AND car_theft_id = ? LIMIT 1");
      $stmt->execute([$userId, $ctId]);
      $cool = $stmt->fetch();
      if ($cool) {
        $left = seconds_left($cool['last_attempt_at'], (int)$ct['cooldown_seconds']);
        if ($left > 0) $error = "Lay low. Try again in " . fmt_time($left) . ".";
      }

      if ($error === '') {
        $roll = random_int(1, 100);
        $baseRate = (int)$ct['success_rate'];
        $rankBonus = (int)($stats['rank_bonus'] ?? 0);
        $pointsNow = (int)($stats['points'] ?? 0);
        $pointsBonus = min(10, intdiv(max(0, $pointsNow), 25));
        $effectiveRate = max(5, min(95, $baseRate + $rankBonus + $pointsBonus + $upgradeBonus));

        $isSuccess = $roll <= $effectiveRate;

        $reward = 0;
        $points = 0;
        $damage = 0;
        $goJail = false;
        $goHospital = false;

        if ($isSuccess) {
          $reward = random_int((int)$ct['min_reward'], (int)$ct['max_reward']);
          $points = random_int((int)$ct['min_points'], (int)$ct['max_points']);
        } else {
          $damage = random_int(4, 14);
          $goJail = (random_int(1,100) <= (int)$ct['jail_chance']);
          $goHospital = (random_int(1,100) <= (int)$ct['hospital_chance']);
          // If both trigger, prefer jail (you can tweak later)
          if ($goJail) $goHospital = false;
        }

        $pdo->beginTransaction();
        try {
          // cooldown upsert
          $stmt = $pdo->prepare("
            INSERT INTO user_car_theft_cooldowns (user_id, car_theft_id, last_attempt_at)
            VALUES (?, ?, NOW())
            ON DUPLICATE KEY UPDATE last_attempt_at = NOW()
          ");
          $stmt->execute([$userId, $ctId]);

          if ($isSuccess) {
            $stmt = $pdo->prepare("UPDATE user_stats SET money = money + ?, points = points + ? WHERE user_id = ?");
            $stmt->execute([$reward, $points, $userId]);
            $logMsg = sprintf("Car theft: %s — Success (+£%d, +%d points).", $ct['name'], $reward, $points);
            $notice = $logMsg;
          } else {
            $stmt = $pdo->prepare("UPDATE user_stats SET health = GREATEST(0, health - ?) WHERE user_id = ?");
            $stmt->execute([$damage, $userId]);

            if ($goJail) {
              $stmt = $pdo->prepare("UPDATE user_stats SET jail_until = DATE_ADD(NOW(), INTERVAL ? SECOND) WHERE user_id = ?");
              $stmt->execute([(int)$ct['jail_seconds'], $userId]);
              $logMsg = sprintf("Car theft: %s — Failed (-%d health) and got arrested (Jail %ds).", $ct['name'], $damage, (int)$ct['jail_seconds']);
            } elseif ($goHospital) {
              $stmt = $pdo->prepare("UPDATE user_stats SET hospital_until = DATE_ADD(NOW(), INTERVAL ? SECOND) WHERE user_id = ?");
              $stmt->execute([(int)$ct['hospital_seconds'], $userId]);
              $logMsg = sprintf("Car theft: %s — Failed (-%d health) and got injured (Hospital %ds).", $ct['name'], $damage, (int)$ct['hospital_seconds']);
            } else {
              $logMsg = sprintf("Car theft: %s — Failed (-%d health).", $ct['name'], $damage);
            }

            $error = $logMsg;
          }

          $stmt = $pdo->prepare("INSERT INTO activity_log (user_id, message) VALUES (?, ?)");
          $stmt->execute([$userId, $logMsg]);

          $pdo->commit();
        } catch (Throwable $e) {
          $pdo->rollBack();
          $error = "Something went wrong. Please try again.";
        }
      }
    }
  }
}

// Load options + cooldowns
$stmt = $pdo->query("SELECT * FROM car_thefts ORDER BY sort_order ASC, id ASC");
$options = $stmt->fetchAll();

$stmt = $pdo->prepare("SELECT car_theft_id, last_attempt_at FROM user_car_theft_cooldowns WHERE user_id = ?");
$stmt->execute([$userId]);
$cooldowns = [];
foreach ($stmt->fetchAll() as $row) {
  $cooldowns[(int)$row['car_theft_id']] = $row['last_attempt_at'];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>Car Theft — State of Crime</title>
  <link rel="stylesheet" href="assets/css/app.css" />
</head>
<body>
  <?php include __DIR__ . '/includes/header.php'; ?>
  <div class="shell">
    <?php include __DIR__ . '/includes/sidebar.php'; ?>
    <main class="content">
      <section class="card">
        <h1 class="h1">Car Theft</h1>
        <p class="muted">Bigger payouts, bigger heat. Success is affected by rank, points, and your Driver Skill upgrade.</p>
        <p class="muted" style="margin-top:8px;">Your modifiers: <strong style="color:#fff;">Rank +<?= (int)($stats['rank_bonus'] ?? 0) ?>%</strong> • <strong style="color:#fff;">Points +<?= min(10, intdiv(max(0, (int)($stats['points'] ?? 0)), 25)) ?>%</strong> • <strong style="color:#fff;">Driver Skill +<?= (int)$upgradeBonus ?>%</strong></p>

        <?php if ($jLeft > 0): ?><div class="alert" style="margin-top:12px;">You are currently in jail for <?= e(fmt_time($jLeft)) ?>. Car theft is disabled. <a href="jail.php" style="color:#fff;">Go to Jail</a></div><?php endif; ?>
        <?php if ($hLeft > 0): ?><div class="alert" style="margin-top:12px;">You are currently in hospital for <?= e(fmt_time($hLeft)) ?>. Car theft is disabled. <a href="hospital.php" style="color:#fff;">Go to Hospital</a></div><?php endif; ?>

        <?php if ($notice): ?><div class="ok" style="margin-top:12px;"><?= e($notice) ?></div><?php endif; ?>
        <?php if ($error): ?><div class="alert" style="margin-top:12px;"><?= e($error) ?></div><?php endif; ?>
      </section>

      <section class="grid" style="margin-top:14px;">
        <?php foreach ($options as $o): ?>
          <?php
            $id = (int)$o['id'];
            $last = $cooldowns[$id] ?? null;
            $cLeft = $last ? seconds_left($last, (int)$o['cooldown_seconds']) : 0;
            $disabled = ($cLeft > 0) || ($jLeft > 0) || ($hLeft > 0);

            $base = (int)$o['success_rate'];
            $eff = max(5, min(95, $base + (int)($stats['rank_bonus'] ?? 0) + min(10, intdiv(max(0, (int)($stats['points'] ?? 0)), 25)) + (int)$upgradeBonus));
          ?>
          <div class="action" style="display:flex;flex-direction:column;gap:10px;">
            <div>
              <div class="action__title"><?= e($o['name']) ?></div>
              <div class="action__desc"><?= e($o['description']) ?></div>
            </div>

            <div class="muted" style="display:flex;flex-wrap:wrap;gap:10px;font-size:.92rem;">
              <span>Success: <strong style="color:#fff;"><?= (int)$eff ?>%</strong></span>
              <span>Reward: <strong style="color:#fff;">£<?= (int)$o['min_reward'] ?>–£<?= (int)$o['max_reward'] ?></strong></span>
              <span>Cooldown: <strong style="color:#fff;"><?= (int)$o['cooldown_seconds'] ?>s</strong></span>
              <span>Jail: <strong style="color:#fff;"><?= (int)$o['jail_chance'] ?>% / <?= (int)$o['jail_seconds'] ?>s</strong></span>
              <span>Hospital: <strong style="color:#fff;"><?= (int)$o['hospital_chance'] ?>% / <?= (int)$o['hospital_seconds'] ?>s</strong></span>
            </div>

            <form method="post" style="margin-top:auto;">
              <input type="hidden" name="csrf_token" value="<?= e(csrf_token()) ?>">
              <input type="hidden" name="car_theft_id" value="<?= $id ?>">
              <button class="btn" type="submit" <?= $disabled ? 'disabled' : '' ?> style="opacity:<?= $disabled ? '0.65' : '1' ?>;">
                <?php if ($jLeft > 0): ?>
                  In Jail: <?= e(fmt_time($jLeft)) ?>
                <?php elseif ($hLeft > 0): ?>
                  In Hospital: <?= e(fmt_time($hLeft)) ?>
                <?php else: ?>
                  <?= $cLeft > 0 ? ('Cooldown: ' . e(fmt_time($cLeft))) : 'Steal Car' ?>
                <?php endif; ?>
              </button>
            </form>
          </div>
        <?php endforeach; ?>
      </section>
    </main>
    <?php include __DIR__ . '/includes/status_panel.php'; ?>
  </div>
  <script src="assets/js/app.js"></script>
</body>
</html>
