(() => {
  const $ = (id) => document.getElementById(id);

  // Mobile menu
  const hamburger = $("hamburger");
  const sidebar = $("sidebar");
  if (hamburger && sidebar) {
    hamburger.addEventListener("click", () => sidebar.classList.toggle("sidebar--open"));
  }

  async function fetchJSON(url) {
    const res = await fetch(url, { headers: { "Accept": "application/json" } });
    return res.json();
  }

  function formatSeconds(s) {
    s = Math.max(0, Number(s) || 0);
    const m = Math.floor(s / 60);
    const r = Math.floor(s % 60);
    if (m > 0) return `${m}m ${String(r).padStart(2,'0')}s`;
    return `${r}s`;
  }

  function formatMoney(n) {
    try {
      return "£" + Number(n).toLocaleString("en-GB");
    } catch {
      return "£" + n;
    }
  }

  async function refreshStatus() {
    try {
      const j = await fetchJSON("api/player_status.php");
      if (!j.ok) return;

      const d = j.data;
      $("statMoney").textContent = formatMoney(d.money);
      $("statBullets").textContent = d.bullets;
      $("statPoints").textContent = d.points;
      $("statRank").textContent = d.rank;
      $("statUnread").textContent = d.unread_messages;
      const hb = $("statHealthBar");
      const ht = $("statHealthText");
      if (ht) ht.textContent = `${d.health} / ${d.max_health}`;
      if (hb) {
        const pct = d.max_health > 0 ? Math.max(0, Math.min(100, (d.health / d.max_health) * 100)) : 0;
        hb.style.width = pct.toFixed(0) + "%";
      }

      const jailEl = $("statJail");
      const hospEl = $("statHospital");
      if (jailEl) {
        const left = Number(d.jail_seconds_left || 0);
        jailEl.textContent = left > 0 ? ("In jail: " + formatSeconds(left)) : "Free";
      }
      if (hospEl) {
        const hleft = Number(d.hospital_seconds_left || 0);
        hospEl.textContent = hleft > 0 ? ("In hospital: " + formatSeconds(hleft)) : "Free";
      }

      // topbar badge
      const badge = $("badgeInbox");
      if (badge) badge.textContent = d.unread_messages;
    } catch (e) {
      // silent
    }
  }

  async function refreshDashboardFeed() {
    const act = $("recentActivity");
    const ann = $("announcements");
    if (!act || !ann) return; // only on dashboard

    try {
      const j = await fetchJSON("api/dashboard_feed.php");
      if (!j.ok) return;

      const { activity, announcements } = j.data;

      act.innerHTML = activity.length
        ? activity.map(x => `<div class="item"><span>${escapeHtml(x.message)}</span><em>${escapeHtml(x.created_at)}</em></div>`).join("")
        : `<div class="muted">No recent activity yet.</div>`;

      ann.innerHTML = announcements.length
        ? announcements.map(x => `<div class="item"><span><strong>${escapeHtml(x.title)}</strong><br>${escapeHtml(x.body)}</span><em>${escapeHtml(x.created_at)}</em></div>`).join("")
        : `<div class="muted">No announcements yet.</div>`;

    } catch (e) {
      // silent
    }
  }

  function escapeHtml(s) {
    return String(s)
      .replaceAll("&", "&amp;")
      .replaceAll("<", "&lt;")
      .replaceAll(">", "&gt;")
      .replaceAll('"', "&quot;")
      .replaceAll("'", "&#039;");
  }

  async function ping() {
    try { await fetch("api/ping.php", { headers: {"Accept":"application/json"}}); } catch(e) {}
  }

  // Initial load
  ping();
  refreshStatus();
  refreshDashboardFeed();

  // Poll
  setInterval(refreshStatus, 7000);
  setInterval(ping, 25000);
  setInterval(refreshDashboardFeed, 15000);
})();
