<?php
// api/jail_action.php
declare(strict_types=1);
require __DIR__ . '/../config.php';
session_start();
require __DIR__ . '/../includes/auth.php';
require_login();

header('Content-Type: application/json; charset=utf-8');

$userId = current_user_id();

function json_out(bool $ok, string $message, array $extra = []): void {
  echo json_encode(array_merge(['ok'=>$ok, 'message'=>$message], $extra));
  exit;
}

csrf_verify();

$action = (string)($_POST['action'] ?? '');

// Load current jail + resources
$stmt = $pdo->prepare("SELECT money, bullets, jail_until, health FROM user_stats WHERE user_id = ? LIMIT 1");
$stmt->execute([$userId]);
$s = $stmt->fetch();
if (!$s) json_out(false, "Stats not found.");

$left = $s['jail_until'] ? max(0, strtotime($s['jail_until']) - time()) : 0;
if ($left <= 0) json_out(false, "You're not in jail.");

$bribeCost = max(250, (int)ceil($left * 4));

if ($action === 'bribe') {
  if ((int)$s['money'] < $bribeCost) json_out(false, "Not enough money. Need £{$bribeCost}.", ['need'=>$bribeCost]);

  $pdo->beginTransaction();
  try {
    $stmt = $pdo->prepare("UPDATE user_stats SET money = money - ?, jail_until = NULL WHERE user_id = ?");
    $stmt->execute([$bribeCost, $userId]);
    $stmt = $pdo->prepare("INSERT INTO activity_log (user_id, message) VALUES (?, ?)");
    $stmt->execute([$userId, "You bribed the guards and got released from jail."]);
    $pdo->commit();
    json_out(true, "Bribe successful. You're free.", ['redirect'=>'dashboard.php']);
  } catch (Throwable $e) {
    $pdo->rollBack();
    json_out(false, "Bribe failed. Try again.");
  }
}

if ($action === 'bustout') {
  $costBullets = 10;
  if ((int)$s['bullets'] < $costBullets) json_out(false, "You need {$costBullets} bullets to attempt a bust-out.");

  $stmt = $pdo->prepare("SELECT points, COALESCE(r.rank_bonus,0) AS rank_bonus FROM user_stats s LEFT JOIN ranks r ON r.id=s.rank_id WHERE s.user_id=? LIMIT 1");
  $stmt->execute([$userId]);
  $st = $stmt->fetch() ?: ['points'=>0,'rank_bonus'=>0];
  $pointsBonus = min(10, intdiv(max(0, (int)$st['points']), 25));
  $chance = min(75, 40 + (int)$st['rank_bonus'] + $pointsBonus);
  $roll = random_int(1,100);
  $success = $roll <= $chance;

  $pdo->beginTransaction();
  try {
    $stmt = $pdo->prepare("UPDATE user_stats SET bullets = GREATEST(0, bullets - ?) WHERE user_id = ?");
    $stmt->execute([$costBullets, $userId]);

    if ($success) {
      $stmt = $pdo->prepare("UPDATE user_stats SET jail_until = NULL WHERE user_id = ?");
      $stmt->execute([$userId]);
      $stmt = $pdo->prepare("INSERT INTO activity_log (user_id, message) VALUES (?, ?)");
      $stmt->execute([$userId, "You organised a bust-out and escaped jail."]);
      $pdo->commit();
      json_out(true, "Bust-out successful. You're free.", ['redirect'=>'dashboard.php']);
    } else {
      $stmt = $pdo->prepare("UPDATE user_stats SET jail_until = DATE_ADD(jail_until, INTERVAL 30 SECOND), health = GREATEST(0, health - 5) WHERE user_id = ?");
      $stmt->execute([$userId]);
      $stmt = $pdo->prepare("INSERT INTO activity_log (user_id, message) VALUES (?, ?)");
      $stmt->execute([$userId, "Bust-out failed. Your sentence got extended."]);
      $pdo->commit();
      // recompute left after extension
      $stmt = $pdo->prepare("SELECT jail_until FROM user_stats WHERE user_id = ?");
      $stmt->execute([$userId]);
      $ju = $stmt->fetch()['jail_until'] ?? null;
      $newLeft = $ju ? max(0, strtotime($ju) - time()) : 0;
      $newBribe = max(250, (int)ceil($newLeft * 4));
      json_out(false, "Bust-out failed. Sentence extended by 30s (-5 health).", ['jail_left'=>$newLeft, 'bribe_cost'=>$newBribe]);
    }
  } catch (Throwable $e) {
    $pdo->rollBack();
    json_out(false, "Bust-out failed. Try again.");
  }
}

json_out(false, "Unknown action.");
